/*
 *
 * stats.c -- Traffic Squeezer Statistics Handler Code
 *
 * Author: Kiran K (2006-2008)
 *         kiran_cyberpro at yahoo dot com
 *
 *
 * Traffic Squeezer is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * Traffic Squeezer is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Zebra; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */


#include "stats.h"
#include "message.h"
#include "web.h"
#include <swill/swill.h>

extern int stats_msgQ_id;
extern char ingress_port[10];
extern char egress_port[10];



void init_stats(ts_stats_t stats)
{
        stats.rx = ZERO;
        stats.tx = ZERO;
        stats.comprsedPkts = ZERO;
        stats.rx_bytes = ZERO;
        stats.tx_bytes = ZERO;
}

void *stats_thread_start(void *a)
{
    key_t  msgkey;
    mymsgbuf qbuf;
    int count=0;
    int log_file_count=0;
 
    init_stats(stats_ingress_port_stats);
    init_stats(stats_egress_port_stats);

    /* create stats receive msgQ */
    {
        /* Generate our IPC key value */
        msgkey = ftok(".", 'm');

        /* Open/create the queue */
        if(( stats_msgQ_id = open_queue( msgkey)) == -1)
        {
              exit(1);
        }
    }

    /*
        rx the message populate the stats
    */

    while(1)
    {   read_message(stats_msgQ_id, MESSAGE_TYPE,  &qbuf );
        count++;
	log_file_count++;
	if(qbuf.msgType == MSG_INGRESS_STATS)
              memcpy( &stats_ingress_port_stats, &qbuf.stats, sizeof(ts_stats_t));
	else if(qbuf.msgType == MSG_EGRESS_STATS)
                memcpy( &stats_egress_port_stats, &qbuf.stats, sizeof(ts_stats_t));

	usleep(10);
        if(count>=5)
        {       system("clear");
                show_intro();
                show_comp_stats(stats_ingress_port_stats, stats_egress_port_stats, ingress_port, egress_port);
                count=0;
                usleep(10);
        }


	/* Record once in a while the snapshot of the compression performance */
	if(log_file_count > LOG_FILE_FREQ) /* Later change this stuff time/interval based also make it configurable  */
	{	FILE *clf; /* compression log file */
		clf = fopen("/var/log/trafficsqueezer/clf.log", "a");
		char in_rx_format='a';
		char in_tx_format='a';
		char eg_rx_format='a';
                char eg_tx_format='a';

                BYTEx2 in_rx_bytes;
		BYTEx2 in_tx_bytes;
		BYTEx2 eg_rx_bytes;
                BYTEx2 eg_tx_bytes;
		
	        in_rx_bytes = human_readable(stats_ingress_port_stats.rx_bytes, &in_rx_format);
		in_tx_bytes = human_readable(stats_ingress_port_stats.tx_bytes, &in_tx_format);
		eg_rx_bytes = human_readable(stats_egress_port_stats.rx_bytes, &eg_rx_format);
		eg_tx_bytes = human_readable(stats_egress_port_stats.tx_bytes, &eg_tx_format);

		//Populate packet count stats
		fprintf(clf, "%u\t%u\t%u\t%u\t", stats_ingress_port_stats.rx, \
					stats_ingress_port_stats.tx, \
					stats_egress_port_stats.rx, \
					stats_egress_port_stats.tx);


		//Populate packet size stats
		fprintf(clf, "%u", in_rx_bytes);
		if(in_rx_format=='a')
			fprintf(clf, "\t");
		else if(in_rx_format=='K')
			fprintf(clf, "000\t");
		else if(in_rx_format=='M')
			fprintf(clf, "000000\t");
		else if(in_rx_format=='G')
			fprintf(clf, "000000000\t");

		fprintf(clf, "%u", in_tx_bytes);
                if(in_tx_format=='a')
                        fprintf(clf, "\t");
                else if(in_tx_format=='K')
                        fprintf(clf, "000\t");
                else if(in_tx_format=='M')
                        fprintf(clf, "000000\t");
                else if(in_tx_format=='G')
                        fprintf(clf, "000000000\t");

		fprintf(clf, "%u", eg_rx_bytes);
                if(eg_rx_format=='a')
                        fprintf(clf, "\t");
                else if(eg_rx_format=='K')
                        fprintf(clf, "000\t");
                else if(eg_rx_format=='M')
                        fprintf(clf, "000000\t");
                else if(eg_rx_format=='G')
                        fprintf(clf, "000000000\t");

		fprintf(clf, "%u", eg_tx_bytes);
                if(eg_tx_format=='a')
                        fprintf(clf, "\n");
                else if(eg_tx_format=='K')
                        fprintf(clf, "000\n");
                else if(eg_tx_format=='M')
                        fprintf(clf, "000000\n");
                else if(eg_tx_format=='G')
                        fprintf(clf, "000000000\n");
		fclose(clf);
		log_file_count = 0;
	}
    }
}







void show_intro()
{
        printf(" Traffic Squeezer (Version: %s, Updated: %s  2006-2008)\n", TRAFFIC_SQUEEZER_VERSION, RELEASE_DATE);
        printf(" Author: Kiran. K\n");
        printf(" URL: http://trafficsqueezer.sourceforge.net\n");


}

void print_performance_bar_ingress(BYTEx16 rx_bytes, BYTEx16 tx_bytes)
{
        int shaded_count=0; /* compressed */
        int unshaded_count=0; /* total actual rx */
        int i,j=0;

        unshaded_count = (( (rx_bytes-tx_bytes)/rx_bytes) *100)/5;
        shaded_count  = ((tx_bytes/rx_bytes)*100)/5;
        printf("[");
        for(i=0;i<shaded_count;i++)
        {
                printf("#");
        }
        for(j=0;j<unshaded_count;j++)
        {
                printf("=");
        }
        printf("]");

}

void print_performance_bar_egress(BYTEx16 rx_bytes, BYTEx16 tx_bytes)
{
        int shaded_count=0; /* compressed */
        int unshaded_count=0; /* total actual rx */
        int i,j=0;

	if(shaded_count>unshaded_count)
        {
	        shaded_count=(100/5)-1;
	}
	else
	{	
        	unshaded_count = (( (tx_bytes-rx_bytes)/tx_bytes) *100)/5;
	        shaded_count  = ((rx_bytes/tx_bytes)*100)/5;
	}


        printf("[");
        for(i=0;i<shaded_count;i++)
        {
                printf("#");
        }
        for(j=0;j<unshaded_count;j++)
        {
                printf("=");
        }
        printf("]");

}

void show_comp_stats(ts_stats_t in_stats, ts_stats_t eg_stats, char *in_port, char *eg_port)
{
	printf("\n Ingress Port: %s\n Egress Port: %s", in_port, eg_port);
        printf("\n Compress Traffic: [%s -> %s]\n", in_port, eg_port);
        printf(" -------------------------------------------------------------------\n");
        printf("  Rx Packets                : %d\n", in_stats.rx);
        printf("  Tx Packets                : %d\n", in_stats.tx);
        printf("  Compressed Packets        : %d\n", in_stats.comprsedPkts);
        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(in_stats.rx_bytes, &format);
                if(format == 'a')
                        printf("  Rx Bytes                  : %u bytes\n", bytes);
                else
                        printf("  Rx Bytes                  : %u %cbytes\n", bytes, format);
        }
        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(in_stats.tx_bytes, &format);
                if(format == 'a')
                        printf("  Tx Bytes                  : %u bytes\n", bytes);
                else
                        printf("  Tx Bytes                  : %u %cbytes\n", bytes, format);

        }

        printf("  Compression Ratio         : "); print_performance_bar_ingress(in_stats.rx_bytes,in_stats.tx_bytes);
        printf("\n");
        printf(" -------------------------------------------------------------------");
        //printf("\n");

        printf("\n De-compress Traffic: [%s -> %s]\n", eg_port, in_port);
        printf(" -------------------------------------------------------------------\n");
        printf("  Rx Packets                : %d\n", eg_stats.rx);
        printf("  Tx Packets                : %d\n", eg_stats.tx);
        printf("  De-compressed Packets     : %d\n", eg_stats.comprsedPkts);
        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(eg_stats.rx_bytes, &format);
                if(format == 'a')
                        printf("  Rx Bytes                  : %u bytes\n", bytes);
                else
                        printf("  Rx Bytes                  : %u %cbytes\n", bytes, format);
        }
        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(eg_stats.tx_bytes, &format);
                if(format == 'a')
                        printf("  Tx Bytes                  : %u bytes\n", bytes);
                else
                        printf("  Tx Bytes                  : %u %cbytes\n", bytes, format);

        }

        printf("  De-compression Ratio      : ");
	print_performance_bar_egress(eg_stats.rx_bytes,eg_stats.tx_bytes);

        printf("\n");
        printf(" -------------------------------------------------------------------\n");
}

void print_html_comp_stats(FILE *f)
{
   //dump_request(stderr);
   BYTEx16 rx_bytes;
   BYTEx16 tx_bytes;
   int shaded_count=0; /* compressed */
   int unshaded_count=0; /* total actual rx */
   int i;
   float in_rx[100], in_tx[100];
   float in_comp_prctge[100];
   float largest_in_bytes = 0;
   float largest_in_comp_prctge = 0;

   float a2[100], b2[100];
   float c2[100];
   float largest_ab2 = 0;

   float largest_c2 = 0;

   int stats_cnt = 0;
    {   char line[110];
        FILE *clf;
        clf = fopen("/var/log/trafficsqueezer/clf.log", "r");
        while( fgets(line, 100, clf) != NULL )
        {       float curr_in_rx=0;
                float curr_in_tx=0;
		float curr_eg_rx=0;
		float curr_eg_tx=0;
		int ignore_packet_rx_tx;
                sscanf(line, "%d\t%d\t%d\t%d\t%f\t%f\t%f\t%f", &ignore_packet_rx_tx, &ignore_packet_rx_tx, \
				&ignore_packet_rx_tx, &ignore_packet_rx_tx, \
				&curr_in_rx, &curr_in_tx, &curr_eg_rx, &curr_eg_tx);

                in_rx[stats_cnt] = curr_in_rx/1000;
                in_tx[stats_cnt] = curr_in_tx/1000;

		if(largest_in_bytes < in_rx[stats_cnt])
			largest_in_bytes = in_rx[stats_cnt];
		if(largest_in_bytes < in_tx[stats_cnt])
                        largest_in_bytes = in_tx[stats_cnt];

		in_comp_prctge[stats_cnt] = ((curr_in_rx - curr_in_tx)/curr_in_rx)*100; /* % of compression ingress port */
		if(largest_in_comp_prctge < in_comp_prctge[stats_cnt])
			largest_in_comp_prctge = in_comp_prctge[stats_cnt];

		a2[stats_cnt] = curr_eg_rx/1000;
                b2[stats_cnt] = curr_eg_tx/1000;

		if(largest_ab2 < a2[stats_cnt])
                        largest_ab2 = a2[stats_cnt];
                if(largest_ab2 < b2[stats_cnt])
                        largest_ab2 = b2[stats_cnt];

                c2[stats_cnt] = ((curr_eg_tx - curr_eg_rx)/curr_eg_tx)*100; /* % of compression egress port */
		if(largest_c2 < c2[stats_cnt])
                        largest_c2 = c2[stats_cnt];


		stats_cnt++;
		if(stats_cnt>99)
			break;

        }

        fclose(clf);
    }

   //web_get_intro_title(f);
   fprintf(f, "<html>\
		<head><meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">\n\
    		<title>Traffic Squeezer Compression Stats</title>\n\
		<link href=\"layout.css\" rel=\"stylesheet\" type=\"text/css\"></link>\n\
    		<!--[if IE]>\
		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/excanvas.pack.js\"></script>\
		<![endif]-->\n\
    		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/jquery.js\"></script>\n\
    		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/jquery.flot.js\"></script>\n\
		</head><body>\
		<img align=left src=\"TS_Reflej_8.png\"> \
            	<table border=0 cellpadding=0 cellspacing=0 \
                align=right vspace=1 style=\"font-family:verdana;font-size:80%\"> \
                <tr><td>Version </td><td> %s</td></tr> \
                <tr><td>Author</td><td> Kiran. K</td><tr><tr><td>Release Date </td><td> %s</td></tr> \
            	<tr><td>Copyright </td><td> 2006-2008</td></tr> \
            	<tr><td>Project Website%s</td><td><a href=\"http://trafficsqueezer.sourceforge.net\"> \
                 http://trafficsqueezer.sourceforge.net</a></td></tr></table> \
             	<br><br><br><br><br><br><hr>\n", TRAFFIC_SQUEEZER_VERSION, RELEASE_DATE, HTTP_TAB);


   fprintf(f, "<font size=\"+1\" face=\"Verdana\" color=AA5088>\
		<center><u><b>Traffic Squeezer - Compression Statistics</b></u></center></font><br>\n");
   
   fprintf(f, "<table border=0 cellpadding=0 cellspacing=0 style=\"font-family:verdana;font-size:90%\">");

   fprintf(f, "<tr><td>Ingress Port</td><td align=left>%s</td></tr> \
	       <tr><td>Egress Port</td><td align=left>%s</td></tr>", ingress_port, egress_port);

   fprintf(f, "<tr> \
		<td height=60 valign=bottom><b><u>Compress Traffic</u>%s [%s -> %s]</b></td> \
		<td></td><td></td> \
		<td height=60 valign=bottom><b><u>De-compress Traffic</u>%s [%s -> %s]</b></td> \
	      </tr>", HTTP_TAB, ingress_port, egress_port, HTTP_TAB, egress_port, ingress_port);

   fprintf(f, "<tr> \
                <td>Rx Packets</td> \
                <td>%d</td><td></td> \
                <td>Rx Packets</td> \
		<td>%d</td></tr>", stats_ingress_port_stats.rx, stats_egress_port_stats.rx );
  fprintf(f, "<tr> \
                <td>Tx Packets</td> \
                <td>%d</td><td></td> \
                <td>Tx Packets</td> \
                <td>%d</td></tr>", stats_ingress_port_stats.tx, stats_egress_port_stats.tx );
  fprintf(f, "<tr> \
                <td>Compressed Packets %s%s</td> \
                <td>%d</td><td></td> \
                <td>De-Compressed Packets %s%s</td> \
                <td>%d</td></tr>", HTTP_TAB, HTTP_TAB, stats_ingress_port_stats.comprsedPkts, \
		HTTP_TAB, HTTP_TAB, stats_egress_port_stats.comprsedPkts );

	{       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_ingress_port_stats.rx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<tr><td>Rx Bytes</td><td>%u bytes</td><td></td>", bytes); 
                else
                fprintf(f, "<tr><td>Rx Bytes</td><td>%u %cbytes</td><td></td>", bytes, format);
        }
	{       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_egress_port_stats.rx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<td>Rx Bytes</td><td>%u bytes</td></tr>", bytes);
                else
                fprintf(f, "<td>Rx Bytes</td><td>%u %cbytes</td></tr>", bytes, format);
        }

        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_ingress_port_stats.tx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<tr><td>Tx Bytes</td><td>%u bytes</td><td></td>", bytes);
                else
                fprintf(f, "<tr><td>Tx Bytes</td><td>%u %cbytes</td><td></td>", bytes, format);
        }
        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_egress_port_stats.tx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<td>Tx Bytes</td><td>%u bytes</td></tr>", bytes);
                else
                fprintf(f, "<td>Tx Bytes</td><td>%u %cbytes</td></tr>", bytes, format);
        }

   fprintf(f, " <tr> \
                <td>Compression Ratio   </td> \
                <td> \
                <table border=1><tr><td><table border=0 cellpadding=0 cellspacing=0> \
                <tr border=0>");

                {       int i;
			char color[7];
			
			rx_bytes = stats_ingress_port_stats.rx_bytes;
                        tx_bytes = stats_ingress_port_stats.tx_bytes;
                        
			shaded_count=0; /* compressed */
                        unshaded_count=0; /* total actual rx */
			if(rx_bytes==0 && tx_bytes==0)
                        {       unshaded_count = 0;
                                shaded_count = 0;
                        }
                        else
                        {
                                unshaded_count = (( (rx_bytes-tx_bytes)/rx_bytes) *100)/5;
                                shaded_count  = ((tx_bytes/rx_bytes)*100)/5;
                        }
			if(shaded_count<=2)
				memcpy(color,"8FD573\0", 7);
			else if(shaded_count>2 && shaded_count<=4)
                                memcpy(color,"B6D573\0", 7);
			else if(shaded_count>4 && shaded_count<=6)
                                memcpy(color,"C8D573\0", 7);
			else if(shaded_count>6 && shaded_count<=8)
                                memcpy(color,"C8D573\0", 7);
			else if(shaded_count>8 && shaded_count<=10)
                                memcpy(color,"D5CA70\0", 7);
                        else if(shaded_count>10 && shaded_count<=12)
                                memcpy(color,"D5C173\0", 7);
                        else if(shaded_count>12 && shaded_count<=14)
                                memcpy(color,"D5A872\0", 7);			
			else if(shaded_count>14 && shaded_count<=16)
                                memcpy(color,"D57272\0", 7);
			else if(shaded_count>16)
				memcpy(color,"D5A880\0", 7);

			if(shaded_count==0 && unshaded_count==0)
                        {       for(i=0;i<19;i++)
                                {       fprintf(f, "<td border=0 width=5 height=10 bgcolor=C1C1C1></td>");
                                }
                        }
			else {
				for(i=0;i<shaded_count;i++)
				{	
					fprintf(f, "<td border=0 width=5 height=10 bgcolor=%s></td>\n", color);
				}
				for(i=0;i<unshaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=555555></td>\n");
                        	}
                        }
                }

                fprintf(f, "</tr></table> \
                </td></tr></table></td> \
	        <td>%s   %s %s %s    %s</td> \
		<td>De-compression Ratio   </td> \
                <td> \
                <table border=1><tr><td><table border=0 cellpadding=0 cellspacing=0> \
                <tr border=0>", HTTP_TAB, HTTP_TAB, HTTP_TAB, HTTP_TAB, HTTP_TAB);

                {       int i;
                        char color[6];

                        rx_bytes = stats_egress_port_stats.rx_bytes;
                        tx_bytes = stats_egress_port_stats.tx_bytes;

                        shaded_count=0; /* de-compressed */
                        unshaded_count=0; /* total actual rx */
			if(rx_bytes==0 && tx_bytes==0)
			{	unshaded_count = 0;
				shaded_count = 0;
			}
			else
			{
                        	unshaded_count = (( (tx_bytes-rx_bytes)/tx_bytes) *100)/5;
                        	shaded_count  = ((rx_bytes/tx_bytes)*100)/5;
                        }
			if(shaded_count<=2)
                                memcpy(color,"8FD573\0", 7);
                        else if(shaded_count>2 && shaded_count<=4)
                                memcpy(color,"B6D573\0", 7);
                        else if(shaded_count>4 && shaded_count<=6)
                                memcpy(color,"C8D573\0", 7);
                        else if(shaded_count>6 && shaded_count<=8)
                                memcpy(color,"C8D573\0", 7);
                        else if(shaded_count>8 && shaded_count<=10)
                                memcpy(color,"D5CA70\0", 7);
                        else if(shaded_count>10 && shaded_count<=12)
                                memcpy(color,"D5C173\0", 7);
                        else if(shaded_count>12 && shaded_count<=14)
                                memcpy(color,"D5A872\0", 7);
                        else if(shaded_count>14 && shaded_count<=16)
                                memcpy(color,"D57272\0", 7);
                        else if(shaded_count>16)
                                memcpy(color,"D5A880\0", 7);

			if(shaded_count==0 && unshaded_count==0)
                        {       for(i=0;i<19;i++)
                                {       fprintf(f, "<td border=0 width=5 height=10 bgcolor=C1C1C1></td>\n");
                                }
                        }
			else
	                {	for(i=0;i<shaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=%s></td>\n", color);
                        	}
                        	for(i=0;i<unshaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=555555></td>\n");
                        	}
			}
                }
   fprintf(f, "</tr></table></td></tr></table></td></tr>\n");
   fprintf(f, "<tr> \
                <td>Compress Traffic Bytes</td> \
                <td><br><br><div id=\"placeholder\" style=\"width:300px;height:150px;\"></div></td><td></td> \
                <td>De-Compress Traffic Bytes</td> \
                <td><br><br><div id=\"placeholder2\" style=\"width:300px;height:150px;\"></div></td></tr>");
   fprintf(f, "<tr> \
                <td>Compress Traffic Ratio</td> \
                <td><br><br><div id=\"placeholder3\" style=\"width:300px;height:150px;\"></div></td><td></td> \
                <td>De-Compress Traffic Ratio</td> \
                <td><br><br><div id=\"placeholder4\" style=\"width:300px;height:150px;\"></div></td></tr>");
   fprintf(f, "</table><br><br>\n", HTTP_TAB, HTTP_TAB);

   fprintf(f, "<br><br><hr>\n");

   fprintf(f, "<table border=0> \
		<tr> \
			<td><center><a href=\"intro.html\"><img border=0 src=\"home.png\"></a></center></td> \
			<td><center><a href=\"settings.html\"><img border=0 src=\"configure.png\"></img></a></center></td> \
			<td><center><a href=\"main_stats.html\"><img border=0 src=\"stats.png\"></img></a></center></td> \
			<td><center><a href=\"help.html\"><img border=0 src=\"help.png\"></a></center></td> \
		</tr> \
		<tr> \
			<td><center>Home</center></td> \
			<td><center>Settings</center></td> \
			<td><center>Statistics</center></td> \
			<td><center>Help</center></td> \
		</tr> \
		</table><br>");

   fprintf(f, "<br><br><br>");
//--------BEGIN------------script
   fprintf(f, "\n<script id=\"source\" language=\"javascript\" type=\"text/javascript\">\n\
	$(function () {\n\
    	    var datasets = {\n\
        	\"in_rx_bytes\": {\n\
            		label: \"Rx (Bytes)\",\n\
            		data: ["); 

   	for(i=0; i<stats_cnt; i++)
	{	if (i==(stats_cnt-1))	
			fprintf(f, "[%d,  %.0f] ", i+1, in_rx[i]);
		else
	 		fprintf(f, "[%d,  %.0f], ", i+1, in_rx[i]);
   	}

   	fprintf(f, "],\n\
            		shadowSize: 0,\n\
            		color: \"#CCCCCC\",\n\
            		//bars: { show: true, fill: true, lineWidth: 1 }\n\
            		lines: { show: true, fill:true, lineWidth: 1 }\n\
        		},\n\
		\n\
		\n\
        	\"in_tx_bytes\": {\n\
            	label: \"Tx (Bytes)\",\n\
            	data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, in_tx[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, in_tx[i]);
   }

        fprintf(f, "],\n\
                 shadowSize: 0,\n\
                 color: \"#EE0000\",\n\
                 //bars: { show: true, fill: true, lineWidth: 1 }\n\
                 lines: { show: true, fill:true, lineWidth: 1 }\n\
                },\n\
                \n\
                \n\
                \"eg_rx_bytes\": {\n\
                label: \"Rx (Bytes)\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, a2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, a2[i]);
   }

        fprintf(f, "],\n\
                        shadowSize: 0,\n\
                        color: \"#CCCCCC\",\n\
                        //bars: { show: true, fill: true, lineWidth: 1 }\n\
                        lines: { show: true, fill:true, lineWidth: 1 }\n\
                        },\n\
                \n\
                \n\
                \"eg_tx_bytes\": {\n\
                label: \"Tx (Bytes)\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, b2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, b2[i]);
   }

   fprintf(f, "],\n\
            	shadowSize: 0,\n\
            	color: \"#EE0000\",\n\
            	//bars: { show: true, fill: true }\n\
            	lines: { show: true, fill:true, lineWidth: 1}\n\
        	},\n\
		\n\
		\n\
	        \"in_comp_ratio\": {\n\
                label: \"Percentage Compression\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, in_comp_prctge[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, in_comp_prctge[i]);
   }


   fprintf(f, "],\n\
                shadowSize: 0,\n\
                color: \"#00aacc\",\n\
                //bars: { show: true, fill: true }\n\
                lines: { show: true, fill:true, lineWidth: 1}\n\
                },\n\
		\n\
                \n\
                \"eg_comp_ratio\": {\n\
                label: \"Percentage De-Compression\",\n\
                data: [ ");

   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, c2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, c2[i]);
   }

   fprintf(f, "],\n\
                shadowSize: 0,\n\
                color: \"#00aacc\",\n\
                //bars: { show: true, fill: true }\n\
                lines: { show: true, fill:true, lineWidth: 1}\n\
                }}\n\
                \n\
                \n\
		\n\
    		function plotAccordingToChoices() {\n\
        	var data1 = [];\n\
        	var data2 = [];\n\
		var data3 = [];\n\
		var data4 = [];\n\
		\n\
		data1.push(datasets[\"in_tx_bytes\"]);\n\
		data1.push(datasets[\"in_rx_bytes\"]);\n\
		data2.push(datasets[\"eg_tx_bytes\"]);\n\
		data2.push(datasets[\"eg_rx_bytes\"]);\n\
		data3.push(datasets[\"in_comp_ratio\"]);\n\
		data4.push(datasets[\"eg_comp_ratio\"]);\n\
		\n\
		\n\
        	//if (data1.length > 0)\n\
            	$.plot($(\"#placeholder\"), data1, {\n\
			grid: { backgroundColor: \"#fefefe\", \n\
			color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
		\n\
		\n\
                yaxis: { //min: 0,\n\
                	ticks: [");
	{       float m;
            	for(m=0; m<largest_in_bytes; m+=(largest_in_bytes/3))
            	{   if(i==0)
			{	fprintf(f, "[0, \" \"], ");
				continue;
			}
			else	    
				fprintf(f, "[%d, \"%dkB\"], ", (int)m, (int)m);
            	}
	        fprintf(f, "[%d, \"%dkB\"]", (int)largest_in_bytes, (int)largest_in_bytes);
	}

   fprintf(f,   "] },\n\
		\n\
		xaxis: { tickDecimals: 0 }\n\
            	});\n\
		\n\
		\n\
		$.plot($(\"#placeholder2\"), data2, {\n\
			grid: { backgroundColor: \"#fefefe\",\n\
			color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
		\n\
		\n\
                yaxis: { //min: 0,\n\
                        ticks: [");

        	{       float m;
	                for(m=0; m<largest_ab2; m+=(largest_ab2/3))
                	{   if(i==0)
                        {       fprintf(f, "[0, \" \"], ");
                                continue;
                        }
                        else
                                fprintf(f, "[%d, \"%dkB\"], ", (int)m, (int)m);
                	}
                	fprintf(f, "[%d, \"%dkB\"]", (int)largest_ab2, (int)largest_ab2);
        	}

   		fprintf(f,   "] },\n\
                \n\
		xaxis: { tickDecimals: 0 }\n\
		});\n\
		\n\
                \n\
                $.plot($(\"#placeholder3\"), data3, {\n\
                        grid: { backgroundColor: \"#fefefe\",\n\
                        color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
                \n\
                \n\
                yaxis: { //min: 0,\n\
                        ticks: [");
		{	float m;
			for(m=0; m<largest_in_comp_prctge; m+=(largest_in_comp_prctge/3))
			{	if(m==0)
                		{       fprintf(f, "[0, \" \"], ");
                        		continue;
                		}
				else
				{	//fprintf(f, "[%d, \"%d\%\"], ", (int)m, (int)m);
				}
			}
			fprintf(f, "[%d, \"%d\%\"]", (int)largest_in_comp_prctge, (int)largest_in_comp_prctge);
		}

   fprintf(f,	"] },\n\
                xaxis: { tickDecimals: 0 }\n\
                });\n\
		\n\
                \n\
                $.plot($(\"#placeholder4\"), data4, {\n\
                        grid: { backgroundColor: \"#fefefe\",\n\
                        color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
                \n\
                \n\
                yaxis: { //min: 0,\n\
                        ticks: [");

                {       float m;
                        for(m=0; m<largest_c2; m+=(largest_c2/3))
                        {       if(m==0)
                                {       fprintf(f, "[0, \" \"], ");
                                        continue;
                                }
                                else
                                {        //fprintf(f, "[%d, \"%d\%\"], ", (int)m, (int)m);
				}
                        }
                        fprintf(f, "[%d, \"%d\%\"]", (int)largest_c2, (int)largest_c2);
                }


  fprintf(f,   "] },\n\
                xaxis: { tickDecimals: 0 }\n\
                });\n\
    	}\n\
    	plotAccordingToChoices();\n\
  });\n\
  </script>\n");

//--------- END -----------script
   fprintf(f, "</body></html>\n");
} /* endof: print_html_comp_stats */




void print_html_coal_stats(FILE *f)
{
   //dump_request(stderr);
   BYTEx16 rx_bytes;
   BYTEx16 tx_bytes;
   int shaded_count=0; /* compressed */
   int unshaded_count=0; /* total actual rx */
   int i;
   float in_rx[100], in_tx[100];
   float in_pkts_prctge[100];
   float largest_in_pkts = 0;
   float largest_in_pkts_prctge = 0;

   float a2[100], b2[100];
   float c2[100];
   float largest_ab2 = 0;

   float largest_c2 = 0;

   int stats_cnt = 0;
    {   char line[110];
        FILE *clf;
        clf = fopen("/var/log/trafficsqueezer/clf.log", "r");
        while( fgets(line, 100, clf) != NULL )
        {       float curr_in_rx=0;
                float curr_in_tx=0;
		float curr_eg_rx=0;
		float curr_eg_tx=0;
		int ignore_bytes_rx_tx;

                sscanf(line, "%f\t%f\t%f\t%f\t%d\t%d\t%d\t%d", &curr_in_rx, &curr_in_tx, &curr_eg_rx, &curr_eg_tx, \
				&ignore_bytes_rx_tx, &ignore_bytes_rx_tx, \
				&ignore_bytes_rx_tx, &ignore_bytes_rx_tx);


                in_rx[stats_cnt] = curr_in_rx;
                in_tx[stats_cnt] = curr_in_tx;

		if(largest_in_pkts < in_rx[stats_cnt])
			largest_in_pkts = in_rx[stats_cnt];
		if(largest_in_pkts < in_tx[stats_cnt])
                        largest_in_pkts = in_tx[stats_cnt];

		in_pkts_prctge[stats_cnt] = ((curr_in_rx - curr_in_tx)/curr_in_rx)*100; /* % of pkts ingress port */
		if(largest_in_pkts_prctge < in_pkts_prctge[stats_cnt])
			largest_in_pkts_prctge = in_pkts_prctge[stats_cnt];

		a2[stats_cnt] = curr_eg_rx;
                b2[stats_cnt] = curr_eg_tx;

		if(largest_ab2 < a2[stats_cnt])
                        largest_ab2 = a2[stats_cnt];
                if(largest_ab2 < b2[stats_cnt])
                        largest_ab2 = b2[stats_cnt];

                c2[stats_cnt] = ((curr_eg_tx - curr_eg_rx)/curr_eg_tx)*100; /* % of pkts egress port */
		if(largest_c2 < c2[stats_cnt])
                        largest_c2 = c2[stats_cnt];


		stats_cnt++;
		if(stats_cnt>99)
			break;

        }

        fclose(clf);
    }

   //web_get_intro_title(f);
   fprintf(f, "<html>\
		<head><meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">\n\
    		<title>Traffic Squeezer Coalescing Stats</title>\n\
		<link href=\"layout.css\" rel=\"stylesheet\" type=\"text/css\"></link>\n\
    		<!--[if IE]>\
		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/excanvas.pack.js\"></script>\
		<![endif]-->\n\
    		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/jquery.js\"></script>\n\
    		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/jquery.flot.js\"></script>\n\
		</head><body>\
		<img align=left src=\"TS_Reflej_8.png\"> \
            	<table border=0 cellpadding=0 cellspacing=0 \
                align=right vspace=1 style=\"font-family:verdana;font-size:80%\"> \
                <tr><td>Version </td><td> %s</td></tr> \
                <tr><td>Author</td><td> Kiran. K</td><tr><tr><td>Release Date </td><td> %s</td></tr> \
            	<tr><td>Copyright </td><td> 2006-2008</td></tr> \
            	<tr><td>Project Website%s</td><td><a href=\"http://trafficsqueezer.sourceforge.net\"> \
                 http://trafficsqueezer.sourceforge.net</a></td></tr></table> \
             	<br><br><br><br><br><br><hr>\n", TRAFFIC_SQUEEZER_VERSION, RELEASE_DATE, HTTP_TAB);


   fprintf(f, "<font size=\"+1\" face=\"Verdana\" color=AA5088>\
		<center><u><b>Traffic Squeezer - Coalescing Statistics</b></u></center></font><br>\n");

   fprintf(f, "<table border=0 cellpadding=0 cellspacing=0 style=\"font-family:verdana;font-size:90%\">");

   fprintf(f, "<tr><td>Ingress Port</td><td align=left>%s</td></tr> \
	       <tr><td>Egress Port</td><td align=left>%s</td></tr>", ingress_port, egress_port);

   fprintf(f, "<tr> \
		<td height=60 valign=bottom><b><u>Ingress Traffic</u>%s [%s -> %s]</b></td> \
		<td></td><td></td> \
		<td height=60 valign=bottom><b><u>Egress Traffic</u>%s [%s -> %s]</b></td> \
	      </tr>", HTTP_TAB, ingress_port, egress_port, HTTP_TAB, egress_port, ingress_port);

   fprintf(f, "<tr> \
                <td>Rx Packets</td> \
                <td>%d</td><td></td> \
                <td>Rx Packets</td> \
		<td>%d</td></tr>", stats_ingress_port_stats.rx, stats_egress_port_stats.rx );
  fprintf(f, "<tr> \
                <td>Tx Packets</td> \
                <td>%d</td><td></td> \
                <td>Tx Packets</td> \
                <td>%d</td></tr>", stats_ingress_port_stats.tx, stats_egress_port_stats.tx );
  fprintf(f, "<tr> \
                <td>Coalesced Packets %s%s</td> \
                <td>%d</td><td></td> \
                <td>Un-Coalesced Packets %s%s</td> \
                <td>%d</td></tr>", HTTP_TAB, HTTP_TAB, stats_ingress_port_stats.coalesedPkts, \
		HTTP_TAB, HTTP_TAB, stats_egress_port_stats.coalesedPkts );

	{       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_ingress_port_stats.rx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<tr><td>Rx Bytes</td><td>%u bytes</td><td></td>", bytes);
                else
                fprintf(f, "<tr><td>Rx Bytes</td><td>%u %cbytes</td><td></td>", bytes, format);
        }
	{       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_egress_port_stats.rx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<td>Rx Bytes</td><td>%u bytes</td></tr>", bytes);
                else
                fprintf(f, "<td>Rx Bytes</td><td>%u %cbytes</td></tr>", bytes, format);
        }

        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_ingress_port_stats.tx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<tr><td>Tx Bytes</td><td>%u bytes</td><td></td>", bytes);
                else
                fprintf(f, "<tr><td>Tx Bytes</td><td>%u %cbytes</td><td></td>", bytes, format);
        }
        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_egress_port_stats.tx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<td>Tx Bytes</td><td>%u bytes</td></tr>", bytes);
                else
                fprintf(f, "<td>Tx Bytes</td><td>%u %cbytes</td></tr>", bytes, format);
        }

#if 0 //Disablling temporarly the Ratio bar in Coalescing
   fprintf(f, " <tr> \
                <td>Compression Ratio   </td> \
                <td> \
                <table border=1><tr><td><table border=0 cellpadding=0 cellspacing=0> \
                <tr border=0>");

                {       int i;
			char color[7];

			rx_bytes = stats_ingress_port_stats.rx_bytes;
                        tx_bytes = stats_ingress_port_stats.tx_bytes;

			shaded_count=0; /* compressed */
                        unshaded_count=0; /* total actual rx */
			if(rx_bytes==0 && tx_bytes==0)
                        {       unshaded_count = 0;
                                shaded_count = 0;
                        }
                        else
                        {
                                unshaded_count = (( (rx_bytes-tx_bytes)/rx_bytes) *100)/5;
                                shaded_count  = ((tx_bytes/rx_bytes)*100)/5;
                        }
			if(shaded_count<=2)
				memcpy(color,"8FD573\0", 7);
			else if(shaded_count>2 && shaded_count<=4)
                                memcpy(color,"B6D573\0", 7);
			else if(shaded_count>4 && shaded_count<=6)
                                memcpy(color,"C8D573\0", 7);
			else if(shaded_count>6 && shaded_count<=8)
                                memcpy(color,"C8D573\0", 7);
			else if(shaded_count>8 && shaded_count<=10)
                                memcpy(color,"D5CA70\0", 7);
                        else if(shaded_count>10 && shaded_count<=12)
                                memcpy(color,"D5C173\0", 7);
                        else if(shaded_count>12 && shaded_count<=14)
                                memcpy(color,"D5A872\0", 7);
			else if(shaded_count>14 && shaded_count<=16)
                                memcpy(color,"D57272\0", 7);
			else if(shaded_count>16)
				memcpy(color,"D5A880\0", 7);

			if(shaded_count==0 && unshaded_count==0)
                        {       for(i=0;i<19;i++)
                                {       fprintf(f, "<td border=0 width=5 height=10 bgcolor=C1C1C1></td>");
                                }
                        }
			else {
				for(i=0;i<shaded_count;i++)
				{
					fprintf(f, "<td border=0 width=5 height=10 bgcolor=%s></td>\n", color);
				}
				for(i=0;i<unshaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=555555></td>\n");
                        	}
                        }
                }

                fprintf(f, "</tr></table> \
                </td></tr></table></td> \
	        <td>%s   %s %s %s    %s</td> \
		<td>De-compression Ratio   </td> \
                <td> \
                <table border=1><tr><td><table border=0 cellpadding=0 cellspacing=0> \
                <tr border=0>", HTTP_TAB, HTTP_TAB, HTTP_TAB, HTTP_TAB, HTTP_TAB);

                {       int i;
                        char color[6];

                        rx_bytes = stats_egress_port_stats.rx_bytes;
                        tx_bytes = stats_egress_port_stats.tx_bytes;

                        shaded_count=0; /* de-compressed */
                        unshaded_count=0; /* total actual rx */
			if(rx_bytes==0 && tx_bytes==0)
			{	unshaded_count = 0;
				shaded_count = 0;
			}
			else
			{
                        	unshaded_count = (( (tx_bytes-rx_bytes)/tx_bytes) *100)/5;
                        	shaded_count  = ((rx_bytes/tx_bytes)*100)/5;
                        }
			if(shaded_count<=2)
                                memcpy(color,"8FD573\0", 7);
                        else if(shaded_count>2 && shaded_count<=4)
                                memcpy(color,"B6D573\0", 7);
                        else if(shaded_count>4 && shaded_count<=6)
                                memcpy(color,"C8D573\0", 7);
                        else if(shaded_count>6 && shaded_count<=8)
                                memcpy(color,"C8D573\0", 7);
                        else if(shaded_count>8 && shaded_count<=10)
                                memcpy(color,"D5CA70\0", 7);
                        else if(shaded_count>10 && shaded_count<=12)
                                memcpy(color,"D5C173\0", 7);
                        else if(shaded_count>12 && shaded_count<=14)
                                memcpy(color,"D5A872\0", 7);
                        else if(shaded_count>14 && shaded_count<=16)
                                memcpy(color,"D57272\0", 7);
                        else if(shaded_count>16)
                                memcpy(color,"D5A880\0", 7);

			if(shaded_count==0 && unshaded_count==0)
                        {       for(i=0;i<19;i++)
                                {       fprintf(f, "<td border=0 width=5 height=10 bgcolor=C1C1C1></td>\n");
                                }
                        }
			else
	                {	for(i=0;i<shaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=%s></td>\n", color);
                        	}
                        	for(i=0;i<unshaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=555555></td>\n");
                        	}
			}
                }
   fprintf(f, "</tr></table></td></tr></table></td></tr>\n");
#endif

   fprintf(f, "<tr> \
                <td>Coalesced Traffic Packets</td> \
                <td><br><br><div id=\"placeholder\" style=\"width:300px;height:150px;\"></div></td><td></td> \
                <td>Un-Coalesced Traffic Packets</td> \
                <td><br><br><div id=\"placeholder2\" style=\"width:300px;height:150px;\"></div></td></tr>");
   fprintf(f, "<tr> \
                <td>Coalesced Traffic Ratio</td> \
                <td><br><br><div id=\"placeholder3\" style=\"width:300px;height:150px;\"></div></td><td></td> \
                <td>Un-Coalesced Traffic Ratio</td> \
                <td><br><br><div id=\"placeholder4\" style=\"width:300px;height:150px;\"></div></td></tr>");
   fprintf(f, "</table><br><br>\n", HTTP_TAB, HTTP_TAB);

   fprintf(f, "<br><br><hr>\n");

   fprintf(f, "<table border=0> \
		<tr> \
			<td><center><a href=\"intro.html\"><img border=0 src=\"home.png\"></a></center></td> \
			<td><center><a href=\"settings.html\"><img border=0 src=\"configure.png\"></img></a></center></td> \
			<td><center><a href=\"main_stats.html\"><img border=0 src=\"stats.png\"></img></a></center></td> \
			<td><center><a href=\"help.html\"><img border=0 src=\"help.png\"></a></center></td> \
		</tr> \
		<tr> \
			<td><center>Home</center></td> \
			<td><center>Settings</center></td> \
			<td><center>Statistics</center></td> \
			<td><center>Help</center></td> \
		</tr> \
		</table><br>");

   fprintf(f, "<br><br><br>");
//--------BEGIN------------script
   fprintf(f, "\n<script id=\"source\" language=\"javascript\" type=\"text/javascript\">\n\
	$(function () {\n\
    	    var datasets = {\n\
        	\"in_rx_pkts\": {\n\
            		label: \"Rx (Packets)\",\n\
            		data: [");

   	for(i=0; i<stats_cnt; i++)
	{	if (i==(stats_cnt-1))
			fprintf(f, "[%d,  %.0f] ", i+1, in_rx[i]);
		else
	 		fprintf(f, "[%d,  %.0f], ", i+1, in_rx[i]);
   	}

   	fprintf(f, "],\n\
            		shadowSize: 0,\n\
            		color: \"#CCCCCC\",\n\
            		//bars: { show: true, fill: true, lineWidth: 1 }\n\
            		lines: { show: true, fill:true, lineWidth: 1 }\n\
        		},\n\
		\n\
		\n\
        	\"in_tx_pkts\": {\n\
            	label: \"Tx (Packets)\",\n\
            	data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, in_tx[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, in_tx[i]);
   }

        fprintf(f, "],\n\
                 shadowSize: 0,\n\
                 color: \"#EE0000\",\n\
                 //bars: { show: true, fill: true, lineWidth: 1 }\n\
                 lines: { show: true, fill:true, lineWidth: 1 }\n\
                },\n\
                \n\
                \n\
                \"eg_rx_pkts\": {\n\
                label: \"Rx (Packets)\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, a2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, a2[i]);
   }

        fprintf(f, "],\n\
                        shadowSize: 0,\n\
                        color: \"#CCCCCC\",\n\
                        //bars: { show: true, fill: true, lineWidth: 1 }\n\
                        lines: { show: true, fill:true, lineWidth: 1 }\n\
                        },\n\
                \n\
                \n\
                \"eg_tx_pkts\": {\n\
                label: \"Tx (Packets)\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, b2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, b2[i]);
   }

   fprintf(f, "],\n\
            	shadowSize: 0,\n\
            	color: \"#EE0000\",\n\
            	//bars: { show: true, fill: true }\n\
            	lines: { show: true, fill:true, lineWidth: 1}\n\
        	},\n\
		\n\
		\n\
	        \"in_coal_ratio\": {\n\
                label: \"Percentage Coalesced\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, in_pkts_prctge[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, in_pkts_prctge[i]);
   }


   fprintf(f, "],\n\
                shadowSize: 0,\n\
                color: \"#00aacc\",\n\
                //bars: { show: true, fill: true }\n\
                lines: { show: true, fill:true, lineWidth: 1}\n\
                },\n\
		\n\
                \n\
                \"eg_coal_ratio\": {\n\
                label: \"Percentage Un-Coalesced\",\n\
                data: [ ");

   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, c2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, c2[i]);
   }

   fprintf(f, "],\n\
                shadowSize: 0,\n\
                color: \"#00aacc\",\n\
                //bars: { show: true, fill: true }\n\
                lines: { show: true, fill:true, lineWidth: 1}\n\
                }}\n\
                \n\
                \n\
		\n\
    		function plotAccordingToChoices() {\n\
        	var data1 = [];\n\
        	var data2 = [];\n\
		var data3 = [];\n\
		var data4 = [];\n\
		\n\
		data1.push(datasets[\"in_tx_pkts\"]);\n\
		data1.push(datasets[\"in_rx_pkts\"]);\n\
		data2.push(datasets[\"eg_tx_pkts\"]);\n\
		data2.push(datasets[\"eg_rx_pkts\"]);\n\
		data3.push(datasets[\"in_coal_ratio\"]);\n\
		data4.push(datasets[\"eg_coal_ratio\"]);\n\
		\n\
		\n\
        	//if (data1.length > 0)\n\
            	$.plot($(\"#placeholder\"), data1, {\n\
			grid: { backgroundColor: \"#fefefe\", \n\
			color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
		\n\
		\n\
                yaxis: { //min: 0,\n\
                	ticks: [");
	{       float m;
            	for(m=0; m<largest_in_pkts; m+=(largest_in_pkts/3))
            	{   if(i==0)
			{	fprintf(f, "[0, \" \"], ");
				continue;
			}
			else
				fprintf(f, "[%d, \"%d\"], ", (int)m, (int)m);
            	}
	        fprintf(f, "[%d, \"%d\"]", (int)largest_in_pkts, (int)largest_in_pkts);
	}

   fprintf(f,   "] },\n\
		\n\
		xaxis: { tickDecimals: 0 }\n\
            	});\n\
		\n\
		\n\
		$.plot($(\"#placeholder2\"), data2, {\n\
			grid: { backgroundColor: \"#fefefe\",\n\
			color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
		\n\
		\n\
                yaxis: { //min: 0,\n\
                        ticks: [");

        	{       float m;
	                for(m=0; m<largest_ab2; m+=(largest_ab2/3))
                	{   if(i==0)
                        {       fprintf(f, "[0, \" \"], ");
                                continue;
                        }
                        else
                                fprintf(f, "[%d, \"%d\"], ", (int)m, (int)m);
                	}
                	fprintf(f, "[%d, \"%d\"]", (int)largest_ab2, (int)largest_ab2);
        	}

   		fprintf(f,   "] },\n\
                \n\
		xaxis: { tickDecimals: 0 }\n\
		});\n\
		\n\
                \n\
                $.plot($(\"#placeholder3\"), data3, {\n\
                        grid: { backgroundColor: \"#fefefe\",\n\
                        color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
                \n\
                \n\
                yaxis: { //min: 0,\n\
                        ticks: [");
		{	float m;
			for(m=0; m<largest_in_pkts_prctge; m+=(largest_in_pkts_prctge/3))
			{	if(m==0)
                		{       fprintf(f, "[0, \" \"], ");
                        		continue;
                		}
				else
				{	//fprintf(f, "[%d, \"%d\%\"], ", (int)m, (int)m);
				}
			}
			fprintf(f, "[%d, \"%d\%\"]", (int)largest_in_pkts_prctge, (int)largest_in_pkts_prctge);
		}

   fprintf(f,	"] },\n\
                xaxis: { tickDecimals: 0 }\n\
                });\n\
		\n\
                \n\
                $.plot($(\"#placeholder4\"), data4, {\n\
                        grid: { backgroundColor: \"#fefefe\",\n\
                        color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
                \n\
                \n\
                yaxis: { //min: 0,\n\
                        ticks: [");

                {       float m;
                        for(m=0; m<largest_c2; m+=(largest_c2/3))
                        {       if(m==0)
                                {       fprintf(f, "[0, \" \"], ");
                                        continue;
                                }
                                else
                                {        //fprintf(f, "[%d, \"%d\%\"], ", (int)m, (int)m);
				}
                        }
                        fprintf(f, "[%d, \"%d\%\"]", (int)largest_c2, (int)largest_c2);
                }


  fprintf(f,   "] },\n\
                xaxis: { tickDecimals: 0 }\n\
                });\n\
    	}\n\
    	plotAccordingToChoices();\n\
  });\n\
  </script>\n");

//--------- END -----------script
   fprintf(f, "</body></html>\n");
} /* endof: print_html_coal_stats */



void print_html_qos_stats(FILE *f)
{
   //dump_request(stderr);
   BYTEx16 rx_bytes;
   BYTEx16 tx_bytes;
   int shaded_count=0; /* compressed */
   int unshaded_count=0; /* total actual rx */
   int i;
   float in_rx[100], in_tx[100];
   float in_comp_prctge[100];
   float largest_in_bytes = 0;
   float largest_in_comp_prctge = 0;

   float a2[100], b2[100];
   float c2[100];
   float largest_ab2 = 0;

   float largest_c2 = 0;

   int stats_cnt = 0;
    {   char line[110];
        FILE *clf;
        clf = fopen("/var/log/trafficsqueezer/clf.log", "r");
        while( fgets(line, 100, clf) != NULL )
        {       float curr_in_rx=0;
                float curr_in_tx=0;
		float curr_eg_rx=0;
		float curr_eg_tx=0;

                sscanf(line, "%f\t%f\t%f\t%f", &curr_in_rx, &curr_in_tx, &curr_eg_rx, &curr_eg_tx);
                in_rx[stats_cnt] = curr_in_rx/1000;
                in_tx[stats_cnt] = curr_in_tx/1000;

		if(largest_in_bytes < in_rx[stats_cnt])
			largest_in_bytes = in_rx[stats_cnt];
		if(largest_in_bytes < in_tx[stats_cnt])
                        largest_in_bytes = in_tx[stats_cnt];

		in_comp_prctge[stats_cnt] = ((curr_in_rx - curr_in_tx)/curr_in_rx)*100; /* % of compression ingress port */
		if(largest_in_comp_prctge < in_comp_prctge[stats_cnt])
			largest_in_comp_prctge = in_comp_prctge[stats_cnt];

		a2[stats_cnt] = curr_eg_rx/1000;
                b2[stats_cnt] = curr_eg_tx/1000;

		if(largest_ab2 < a2[stats_cnt])
                        largest_ab2 = a2[stats_cnt];
                if(largest_ab2 < b2[stats_cnt])
                        largest_ab2 = b2[stats_cnt];

                c2[stats_cnt] = ((curr_eg_tx - curr_eg_rx)/curr_eg_tx)*100; /* % of compression egress port */
		if(largest_c2 < c2[stats_cnt])
                        largest_c2 = c2[stats_cnt];


		stats_cnt++;
		if(stats_cnt>99)
			break;

        }

        fclose(clf);
    }

   //web_get_intro_title(f);
   fprintf(f, "<html>\
		<head><meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">\n\
    		<title>Traffic Squeezer QoS Stats</title>\n\
		<link href=\"layout.css\" rel=\"stylesheet\" type=\"text/css\"></link>\n\
    		<!--[if IE]>\
		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/excanvas.pack.js\"></script>\
		<![endif]-->\n\
    		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/jquery.js\"></script>\n\
    		<script language=\"javascript\" type=\"text/javascript\" src=\"flot/jquery.flot.js\"></script>\n\
		</head><body>\
		<img align=left src=\"TS_Reflej_8.png\"> \
            	<table border=0 cellpadding=0 cellspacing=0 \
                align=right vspace=1 style=\"font-family:verdana;font-size:80%\"> \
                <tr><td>Version </td><td> %s</td></tr> \
                <tr><td>Author</td><td> Kiran. K</td><tr><tr><td>Release Date </td><td> %s</td></tr> \
            	<tr><td>Copyright </td><td> 2006-2008</td></tr> \
            	<tr><td>Project Website%s</td><td><a href=\"http://trafficsqueezer.sourceforge.net\"> \
                 http://trafficsqueezer.sourceforge.net</a></td></tr></table> \
             	<br><br><br><br><br><br><hr>\n", TRAFFIC_SQUEEZER_VERSION, RELEASE_DATE, HTTP_TAB);


   fprintf(f, "<font size=\"+1\" face=\"Verdana\" color=AA5088>\
		<center><u><b>Traffic Squeezer - QoS Statistics</b></u></center></font><br>\n");
   
   fprintf(f, "<table border=0 cellpadding=0 cellspacing=0 style=\"font-family:verdana;font-size:90%\">");

   fprintf(f, "<tr><td>Ingress Port</td><td align=left>%s</td></tr> \
	       <tr><td>Egress Port</td><td align=left>%s</td></tr>", ingress_port, egress_port);

   fprintf(f, "<tr> \
		<td height=60 valign=bottom><b><u>Compress Traffic</u>%s [%s -> %s]</b></td> \
		<td></td><td></td> \
		<td height=60 valign=bottom><b><u>De-compress Traffic</u>%s [%s -> %s]</b></td> \
	      </tr>", HTTP_TAB, ingress_port, egress_port, HTTP_TAB, egress_port, ingress_port);

   fprintf(f, "<tr> \
                <td>Rx Packets</td> \
                <td>%d</td><td></td> \
                <td>Rx Packets</td> \
		<td>%d</td></tr>", stats_ingress_port_stats.rx, stats_egress_port_stats.rx );
  fprintf(f, "<tr> \
                <td>Tx Packets</td> \
                <td>%d</td><td></td> \
                <td>Tx Packets</td> \
                <td>%d</td></tr>", stats_ingress_port_stats.tx, stats_egress_port_stats.tx );
  fprintf(f, "<tr> \
                <td>Compressed Packets %s%s</td> \
                <td>%d</td><td></td> \
                <td>De-Compressed Packets %s%s</td> \
                <td>%d</td></tr>", HTTP_TAB, HTTP_TAB, stats_ingress_port_stats.comprsedPkts, \
		HTTP_TAB, HTTP_TAB, stats_egress_port_stats.comprsedPkts );

	{       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_ingress_port_stats.rx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<tr><td>Rx Bytes</td><td>%u bytes</td><td></td>", bytes); 
                else
                fprintf(f, "<tr><td>Rx Bytes</td><td>%u %cbytes</td><td></td>", bytes, format);
        }
	{       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_egress_port_stats.rx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<td>Rx Bytes</td><td>%u bytes</td></tr>", bytes);
                else
                fprintf(f, "<td>Rx Bytes</td><td>%u %cbytes</td></tr>", bytes, format);
        }

        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_ingress_port_stats.tx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<tr><td>Tx Bytes</td><td>%u bytes</td><td></td>", bytes);
                else
                fprintf(f, "<tr><td>Tx Bytes</td><td>%u %cbytes</td><td></td>", bytes, format);
        }
        {       char format='a';
                BYTEx2 bytes;
                bytes = human_readable(stats_egress_port_stats.tx_bytes, &format);
                if(format == 'a')
                        fprintf(f, "<td>Tx Bytes</td><td>%u bytes</td></tr>", bytes);
                else
                fprintf(f, "<td>Tx Bytes</td><td>%u %cbytes</td></tr>", bytes, format);
        }

   fprintf(f, " <tr> \
                <td>Compression Ratio   </td> \
                <td> \
                <table border=1><tr><td><table border=0 cellpadding=0 cellspacing=0> \
                <tr border=0>");

                {       int i;
			char color[7];
			
			rx_bytes = stats_ingress_port_stats.rx_bytes;
                        tx_bytes = stats_ingress_port_stats.tx_bytes;
                        
			shaded_count=0; /* compressed */
                        unshaded_count=0; /* total actual rx */
			if(rx_bytes==0 && tx_bytes==0)
                        {       unshaded_count = 0;
                                shaded_count = 0;
                        }
                        else
                        {
                                unshaded_count = (( (rx_bytes-tx_bytes)/rx_bytes) *100)/5;
                                shaded_count  = ((tx_bytes/rx_bytes)*100)/5;
                        }
			if(shaded_count<=2)
				memcpy(color,"8FD573\0", 7);
			else if(shaded_count>2 && shaded_count<=4)
                                memcpy(color,"B6D573\0", 7);
			else if(shaded_count>4 && shaded_count<=6)
                                memcpy(color,"C8D573\0", 7);
			else if(shaded_count>6 && shaded_count<=8)
                                memcpy(color,"C8D573\0", 7);
			else if(shaded_count>8 && shaded_count<=10)
                                memcpy(color,"D5CA70\0", 7);
                        else if(shaded_count>10 && shaded_count<=12)
                                memcpy(color,"D5C173\0", 7);
                        else if(shaded_count>12 && shaded_count<=14)
                                memcpy(color,"D5A872\0", 7);			
			else if(shaded_count>14 && shaded_count<=16)
                                memcpy(color,"D57272\0", 7);
			else if(shaded_count>16)
				memcpy(color,"D5A880\0", 7);

			if(shaded_count==0 && unshaded_count==0)
                        {       for(i=0;i<19;i++)
                                {       fprintf(f, "<td border=0 width=5 height=10 bgcolor=C1C1C1></td>");
                                }
                        }
			else {
				for(i=0;i<shaded_count;i++)
				{	
					fprintf(f, "<td border=0 width=5 height=10 bgcolor=%s></td>\n", color);
				}
				for(i=0;i<unshaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=555555></td>\n");
                        	}
                        }
                }

                fprintf(f, "</tr></table> \
                </td></tr></table></td> \
	        <td>%s   %s %s %s    %s</td> \
		<td>De-compression Ratio   </td> \
                <td> \
                <table border=1><tr><td><table border=0 cellpadding=0 cellspacing=0> \
                <tr border=0>", HTTP_TAB, HTTP_TAB, HTTP_TAB, HTTP_TAB, HTTP_TAB);

                {       int i;
                        char color[6];

                        rx_bytes = stats_egress_port_stats.rx_bytes;
                        tx_bytes = stats_egress_port_stats.tx_bytes;

                        shaded_count=0; /* de-compressed */
                        unshaded_count=0; /* total actual rx */
			if(rx_bytes==0 && tx_bytes==0)
			{	unshaded_count = 0;
				shaded_count = 0;
			}
			else
			{
                        	unshaded_count = (( (tx_bytes-rx_bytes)/tx_bytes) *100)/5;
                        	shaded_count  = ((rx_bytes/tx_bytes)*100)/5;
                        }
			if(shaded_count<=2)
                                memcpy(color,"8FD573\0", 7);
                        else if(shaded_count>2 && shaded_count<=4)
                                memcpy(color,"B6D573\0", 7);
                        else if(shaded_count>4 && shaded_count<=6)
                                memcpy(color,"C8D573\0", 7);
                        else if(shaded_count>6 && shaded_count<=8)
                                memcpy(color,"C8D573\0", 7);
                        else if(shaded_count>8 && shaded_count<=10)
                                memcpy(color,"D5CA70\0", 7);
                        else if(shaded_count>10 && shaded_count<=12)
                                memcpy(color,"D5C173\0", 7);
                        else if(shaded_count>12 && shaded_count<=14)
                                memcpy(color,"D5A872\0", 7);
                        else if(shaded_count>14 && shaded_count<=16)
                                memcpy(color,"D57272\0", 7);
                        else if(shaded_count>16)
                                memcpy(color,"D5A880\0", 7);

			if(shaded_count==0 && unshaded_count==0)
                        {       for(i=0;i<19;i++)
                                {       fprintf(f, "<td border=0 width=5 height=10 bgcolor=C1C1C1></td>\n");
                                }
                        }
			else
	                {	for(i=0;i<shaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=%s></td>\n", color);
                        	}
                        	for(i=0;i<unshaded_count;i++)
                        	{
                                	fprintf(f, "<td border=0 width=5 height=10 bgcolor=555555></td>\n");
                        	}
			}
                }
   fprintf(f, "</tr></table></td></tr></table></td></tr>\n");
   fprintf(f, "<tr> \
                <td>Compress Traffic Bytes</td> \
                <td><br><br><div id=\"placeholder\" style=\"width:300px;height:150px;\"></div></td><td></td> \
                <td>De-Compress Traffic Bytes</td> \
                <td><br><br><div id=\"placeholder2\" style=\"width:300px;height:150px;\"></div></td></tr>");
   fprintf(f, "<tr> \
                <td>Compress Traffic Ratio</td> \
                <td><br><br><div id=\"placeholder3\" style=\"width:300px;height:150px;\"></div></td><td></td> \
                <td>De-Compress Traffic Ratio</td> \
                <td><br><br><div id=\"placeholder4\" style=\"width:300px;height:150px;\"></div></td></tr>");
   fprintf(f, "</table><br><br>\n", HTTP_TAB, HTTP_TAB);

   fprintf(f, "<br><br><hr>\n");

   fprintf(f, "<table border=0> \
		<tr> \
			<td><center><a href=\"intro.html\"><img border=0 src=\"home.png\"></a></center></td> \
			<td><center><a href=\"settings.html\"><img border=0 src=\"configure.png\"></img></a></center></td> \
			<td><center><a href=\"main_stats.html\"><img border=0 src=\"stats.png\"></img></a></center></td> \
			<td><center><a href=\"help.html\"><img border=0 src=\"help.png\"></a></center></td> \
		</tr> \
		<tr> \
			<td><center>Home</center></td> \
			<td><center>Settings</center></td> \
			<td><center>Statistics</center></td> \
			<td><center>Help</center></td> \
		</tr> \
		</table><br>");

   fprintf(f, "<br><br><br>");
//--------BEGIN------------script
   fprintf(f, "\n<script id=\"source\" language=\"javascript\" type=\"text/javascript\">\n\
	$(function () {\n\
    	    var datasets = {\n\
        	\"in_rx_bytes\": {\n\
            		label: \"Rx (Bytes)\",\n\
            		data: ["); 

   	for(i=0; i<stats_cnt; i++)
	{	if (i==(stats_cnt-1))	
			fprintf(f, "[%d,  %.0f] ", i+1, in_rx[i]);
		else
	 		fprintf(f, "[%d,  %.0f], ", i+1, in_rx[i]);
   	}

   	fprintf(f, "],\n\
            		shadowSize: 0,\n\
            		color: \"#CCCCCC\",\n\
            		//bars: { show: true, fill: true, lineWidth: 1 }\n\
            		lines: { show: true, fill:true, lineWidth: 1 }\n\
        		},\n\
		\n\
		\n\
        	\"in_tx_bytes\": {\n\
            	label: \"Tx (Bytes)\",\n\
            	data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, in_tx[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, in_tx[i]);
   }

        fprintf(f, "],\n\
                 shadowSize: 0,\n\
                 color: \"#EE0000\",\n\
                 //bars: { show: true, fill: true, lineWidth: 1 }\n\
                 lines: { show: true, fill:true, lineWidth: 1 }\n\
                },\n\
                \n\
                \n\
                \"eg_rx_bytes\": {\n\
                label: \"Rx (Bytes)\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, a2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, a2[i]);
   }

        fprintf(f, "],\n\
                        shadowSize: 0,\n\
                        color: \"#CCCCCC\",\n\
                        //bars: { show: true, fill: true, lineWidth: 1 }\n\
                        lines: { show: true, fill:true, lineWidth: 1 }\n\
                        },\n\
                \n\
                \n\
                \"eg_tx_bytes\": {\n\
                label: \"Tx (Bytes)\",\n\
                data: [ ");


   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, b2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, b2[i]);
   }

   fprintf(f, "],\n\
            	shadowSize: 0,\n\
            	color: \"#EE0000\",\n\
            	//bars: { show: true, fill: true }\n\
            	lines: { show: true, fill:true, lineWidth: 1}\n\
        	},\n\
		\n\
		\n\
	        \"in_comp_ratio\": {\n\
                label: \"Percentage Compression\",\n\
                data: [ ");
			

   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, in_comp_prctge[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, in_comp_prctge[i]);
   }


   fprintf(f, "],\n\
                shadowSize: 0,\n\
                color: \"#00aacc\",\n\
                //bars: { show: true, fill: true }\n\
                lines: { show: true, fill:true, lineWidth: 1}\n\
                },\n\
		\n\
                \n\
                \"eg_comp_ratio\": {\n\
                label: \"Percentage De-Compression\",\n\
                data: [ ");

   for(i=0; i<stats_cnt; i++)
   {    if (i==(stats_cnt-1))
                fprintf(f, "[%d,  %.0f] ", i+1, c2[i]);
        else
                fprintf(f, "[%d,  %.0f], ", i+1, c2[i]);
   }

   fprintf(f, "],\n\
                shadowSize: 0,\n\
                color: \"#00aacc\",\n\
                //bars: { show: true, fill: true }\n\
                lines: { show: true, fill:true, lineWidth: 1}\n\
                }}\n\
                \n\
                \n\
		\n\
    		function plotAccordingToChoices() {\n\
        	var data1 = [];\n\
        	var data2 = [];\n\
		var data3 = [];\n\
		var data4 = [];\n\
		\n\
		data1.push(datasets[\"in_tx_bytes\"]);\n\
		data1.push(datasets[\"in_rx_bytes\"]);\n\
		data2.push(datasets[\"eg_tx_bytes\"]);\n\
		data2.push(datasets[\"eg_rx_bytes\"]);\n\
		data3.push(datasets[\"in_comp_ratio\"]);\n\
		data4.push(datasets[\"eg_comp_ratio\"]);\n\
		\n\
		\n\
        	//if (data1.length > 0)\n\
            	$.plot($(\"#placeholder\"), data1, {\n\
			grid: { backgroundColor: \"#fefefe\", \n\
			color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
		\n\
		\n\
                yaxis: { //min: 0,\n\
                	ticks: [");
	{       float m;
            	for(m=0; m<largest_in_bytes; m+=(largest_in_bytes/3))
            	{   if(i==0)
			{	fprintf(f, "[0, \" \"], ");
				continue;
			}
			else	    
				fprintf(f, "[%d, \"%dkB\"], ", (int)m, (int)m);
            	}
	        fprintf(f, "[%d, \"%dkB\"]", (int)largest_in_bytes, (int)largest_in_bytes);
	}

   fprintf(f,   "] },\n\
		\n\
		xaxis: { tickDecimals: 0 }\n\
            	});\n\
		\n\
		\n\
		$.plot($(\"#placeholder2\"), data2, {\n\
			grid: { backgroundColor: \"#fefefe\",\n\
			color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
		\n\
		\n\
                yaxis: { //min: 0,\n\
                        ticks: [");

        	{       float m;
	                for(m=0; m<largest_ab2; m+=(largest_ab2/3))
                	{   if(i==0)
                        {       fprintf(f, "[0, \" \"], ");
                                continue;
                        }
                        else
                                fprintf(f, "[%d, \"%dkB\"], ", (int)m, (int)m);
                	}
                	fprintf(f, "[%d, \"%dkB\"]", (int)largest_ab2, (int)largest_ab2);
        	}

   		fprintf(f,   "] },\n\
                \n\
		xaxis: { tickDecimals: 0 }\n\
		});\n\
		\n\
                \n\
                $.plot($(\"#placeholder3\"), data3, {\n\
                        grid: { backgroundColor: \"#fefefe\",\n\
                        color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
                \n\
                \n\
                yaxis: { //min: 0,\n\
                        ticks: [");
		{	float m;
			for(m=0; m<largest_in_comp_prctge; m+=(largest_in_comp_prctge/3))
			{	if(m==0)
                		{       fprintf(f, "[0, \" \"], ");
                        		continue;
                		}
				else
				{	//fprintf(f, "[%d, \"%d\%\"], ", (int)m, (int)m);
				}
			}
			fprintf(f, "[%d, \"%d\%\"]", (int)largest_in_comp_prctge, (int)largest_in_comp_prctge);
		}

   fprintf(f,	"] },\n\
                xaxis: { tickDecimals: 0 }\n\
                });\n\
		\n\
                \n\
                $.plot($(\"#placeholder4\"), data4, {\n\
                        grid: { backgroundColor: \"#fefefe\",\n\
                        color: \"#010101\", tickColor: \"#f1f1f1\", labelMargin: 0 },\n\
                \n\
                \n\
                yaxis: { //min: 0,\n\
                        ticks: [");

                {       float m;
                        for(m=0; m<largest_c2; m+=(largest_c2/3))
                        {       if(m==0)
                                {       fprintf(f, "[0, \" \"], ");
                                        continue;
                                }
                                else
                                {        //fprintf(f, "[%d, \"%d\%\"], ", (int)m, (int)m);
				}
                        }
                        fprintf(f, "[%d, \"%d\%\"]", (int)largest_c2, (int)largest_c2);
                }


  fprintf(f,   "] },\n\
                xaxis: { tickDecimals: 0 }\n\
                });\n\
    	}\n\
    	plotAccordingToChoices();\n\
  });\n\
  </script>\n");

//--------- END -----------script
   fprintf(f, "</body></html>\n");
} /* endof: print_html_qos_stats */

